
// Avisynth v2.5.  Copyright 2002 Ben Rudiak-Gould et al.
// http://www.avisynth.org

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
// http://www.gnu.org/copyleft/gpl.html .
//
// Linking Avisynth statically or dynamically with other modules is making a
// combined work based on Avisynth.  Thus, the terms and conditions of the GNU
// General Public License cover the whole combination.
//
// As a special exception, the copyright holders of Avisynth give you
// permission to link Avisynth with independent modules that communicate with
// Avisynth solely through the interfaces defined in avisynth.h, regardless of the license
// terms of these independent modules, and to copy and distribute the
// resulting combined work under terms of your choice, provided that
// every copy of the combined work is accompanied by a complete copy of
// the source code of Avisynth (the version of Avisynth used to produce the
// combined work), being distributed under the terms of the GNU General
// Public License plus this exception.  An independent module is a module
// which is not derived from or based on Avisynth, such as 3rd-party filters,
// import and export plugins, or graphical user interfaces.

#include "conditional_audio.h"
#include <float.h>
#include <math.h>        // log() p
#include "avisynth.h" 

AVSValue MinMaxAudio::MinMaxA(AVSValue clip, void* user_data, int channel, int mode, IScriptEnvironment* env) {
  
  if (!clip.IsClip())
    env->ThrowError("MinMaxA: No clip supplied!");

  PClip child = clip.AsClip();
  VideoInfo vi = child->GetVideoInfo();
  
  if (!vi.HasAudio())
    env->ThrowError("MinMaxAudio: Clip contains no audio!");

  // Get current frame number
  AVSValue cf = env->GetVar("current_frame");
  if (!cf.IsInt())
 	env->ThrowError("MinMaxAudio: This filter can only be used within ConditionalFilter");

  int n = cf.AsInt();

  PVideoFrame src = child->GetFrame(n, env);

  AVSValue cache_args[1] = { child };
  PClip aud_clip = env->Invoke("ConvertAudioToFloat", AVSValue(cache_args,1)).AsClip();

  const int channels = vi.AudioChannels();

  // check that 0<=channel<=channels
  if (!((channel>=0)&&(channel<=channels)))
    env->ThrowError("MinMaxAudio: Make sure that 0<=channel<=%d!", channels);

  // Get audio for current frame.
  const __int64 start = vi.AudioSamplesFromFrames(n);
  const __int64 count = vi.AudioSamplesFromFrames(1);
  const int c = count*channels;

  SFLOAT* samples = new SFLOAT[c];
  aud_clip->GetAudio(samples, max(0,start), count, env);

  const int first = (channel == 0) ? 0 : channel-1;
  const int incr  = (channel == 0) ? 1 : channels;
  const int tot   = (channel == 0) ? c : (int)count;

  AVSValue Result = AVSValue();
  // Find minimal volume. This is mostly useless as audio is AC
  // and we are just finding the lowest near zero crossing. It might
  // just be usefull with very high fps and low frequency sounds.
  if (mode == MIN) {
    float min_vol = FLT_MAX; // from <float.h>
    for (int i=first; i < c; i+=incr)
      min_vol = min(min_vol, fabs(samples[i]));

    double db = (min_vol==0.0f) ? -759.0 : 8.685889638 * log(min_vol);
	// 20*log10(FLT_MIN) = 20*log10(1.175e-38) = -759
    Result = db;
  }
  // Find maximal volume.
  if (mode == MAX) {
    float max_vol = 0.0f;
    for (int i=first; i < c; i+=incr)
      max_vol = max(max_vol, fabs(samples[i]));

    double db = (max_vol==0.0f) ? -759.0 : 8.685889638 * log(max_vol); // 20*log10(max)
	// 20*log10(FLT_MIN) = 20*log10(1.175e-38) = -759
    Result = db;
  }
  // Find rms.
  if (mode == RMS) {
    double rms_vol = 0.0;
    for (int i=first; i < c; i+=incr) {
      const double sample = samples[i];
      rms_vol += sample*sample;
    }
    rms_vol = (rms_vol==0.0) ? -791.0 : 8.685889638/2. * log(rms_vol/tot); // 20*log10(sqrt(ms))
	// 20*log10(FLT_MIN/tot) = 20*log10(c*1.175e-38/c) = xxx
    Result = rms_vol;
  }
  delete[] samples; // don't leak!
  return Result;
}


AVSValue __cdecl MinMaxAudio::Create_min_audio(AVSValue args, void* user_data, IScriptEnvironment* env) {
	return MinMaxA(args[0], user_data, args[1].AsInt(0), MIN, env);
}

AVSValue __cdecl MinMaxAudio::Create_max_audio(AVSValue args, void* user_data, IScriptEnvironment* env) {
	return MinMaxA(args[0], user_data, args[1].AsInt(0), MAX, env);
}

AVSValue __cdecl MinMaxAudio::Create_rms_audio(AVSValue args, void* user_data, IScriptEnvironment* env) {
	return MinMaxA(args[0], user_data, args[1].AsInt(0), RMS, env);
}

extern "C" __declspec(dllexport) const char* __stdcall AvisynthPluginInit2(IScriptEnvironment* env) {
    env->AddFunction("AudioMin", "c[channel]i", MinMaxAudio::Create_min_audio, 0);
    env->AddFunction("AudioMax", "c[channel]i", MinMaxAudio::Create_max_audio, 0);
    env->AddFunction("AudioRMS", "c[channel]i", MinMaxAudio::Create_rms_audio, 0);
    return "`MinMaxAudio' MinMaxAudio plugin";
}


const AVS_Linkage* AVS_linkage = nullptr;

extern "C" __declspec(dllexport)
const char* __stdcall AvisynthPluginInit3(IScriptEnvironment* env,
    const AVS_Linkage* const vectors) {
    AVS_linkage = vectors;   // dvF֐e[uۑ
    //  Init2 ̂܂܌Ăœo^ėp
    return AvisynthPluginInit2(env);
}